# ============================ Treap ==============================
#

from typing import Tuple, NamedTuple
import random, sys

MAXPRI = 2**20
random.seed(47)


class Treap(NamedTuple):
    key: int  # values are sorted from left to right by key; !!! must be an integer
    prio: int
    my_weight: int  # total weight of subtree incl myself
    weight: int  # total weight of subtree incl myself
    neg_sum: int  # total sum of negative keys of subtree incl myself
    pos_sum: int  # total sum of positive keys of subtree incl myself
    left: "Treap"
    right: "Treap"


def new_treap(
    key: int, prio: int, my_weight: int, left: Treap = None, right: Treap = None
) -> Treap:
    weight = my_weight + (bool(left) and left.weight) + (bool(right) and right.weight)
    neg_sum = (
        my_weight * (key * (key < 0))
        + (bool(left) and left.neg_sum)
        + (bool(right) and right.neg_sum)
    )
    pos_sum = (
        my_weight * (key * (key > 0))
        + (bool(left) and left.pos_sum)
        + (bool(right) and right.pos_sum)
    )
    return Treap(key, prio, my_weight, weight, neg_sum, pos_sum, left, right)


def split(T: Treap, key: int) -> Tuple[Treap, Treap]:
    if T is None:
        return None, None
    if T.key <= key:
        L, R = split(T.right, key)
        return new_treap(T.key, T.prio, T.my_weight, T.left, L), R
    else:
        L, R = split(T.left, key)
        return L, new_treap(T.key, T.prio, T.my_weight, R, T.right)


def merge(T1: Treap, T2: Treap) -> Treap:
    if T2 is None:
        return T1
    if T1 is None:
        return T2
    if T1.prio > T2.prio:
        return new_treap(T1.key, T1.prio, T1.my_weight, T1.left, merge(T1.right, T2))
    else:
        return new_treap(T2.key, T2.prio, T2.my_weight, merge(T1, T2.left), T2.right)


fmt = lambda T: (
    "-" if T is None else f"({fmt(T.left)} ({T.key}, {T.my_weight}) {fmt(T.right)})"
)
ofmt = lambda T: (
    "" if T is None else f"{ofmt(T.left)}({T.key}, {T.my_weight}), {ofmt(T.right)}"
)


class SortedList:
    def __init__(self):
        self.root = None
        self.sum = 0

    def add(self, value: int, weight: int = 1):
        self.sum += value * weight
        T1, T2 = split(self.root, value)
        T1, k = split(T1, value - 1)
        if k:
            assert k.key == value, "bug in code"
            weight += k.my_weight
            prio = k.prio
        else:
            prio = random.randint(0, MAXPRI)
        self.root = merge(T1, merge(new_treap(value, prio, weight), T2))

    def remove(self, value: int, weight: int = 1):
        self.sum -= value * weight
        T1, T2 = split(self.root, value)
        T1, k = split(T1, value - 1)
        assert (
            k.key == value and k.my_weight >= weight
        ), "you can only remove existing elements"
        if k.weight == weight:
            self.root = merge(T1, T2)
        else:
            self.root = merge(
                T1, merge(new_treap(value, k.prio, k.my_weight - weight), T2)
            )

    def sum_smallest_k_negatives(self, k, neg=True) -> int:
        T = self.root
        skipped_sum = 0
        while T:
            left_weight = bool(T.left) and T.left.weight
            left_sum = bool(T.left) and (T.left.neg_sum if neg else T.left.pos_sum)
            my_sum = ((neg and T.key < 0) or ((not neg) and T.key > 0)) * T.key
            # print(k, fmt(T), T.my_weight, left_weight, left_sum, skipped_sum)

            if k < left_weight:
                T = T.left
            elif k <= left_weight + T.my_weight:
                return skipped_sum + left_sum + my_sum * (k - left_weight)
            else:
                k -= left_weight + T.my_weight
                skipped_sum += left_sum + my_sum * T.my_weight
                T = T.right
        raise KeyError("something weird happened")

    def sum_largest_k_positives(self, k) -> int:
        return self.root.pos_sum - self.sum_smallest_k_negatives(
            self.root.weight - k, neg=False
        )

# ============================ The actual solution ==============================

N, P, K = [int(x) for x in sys.stdin.readline().split()]
D = []
for n in range(N):
    smer, vzd = sys.stdin.readline().split()
    D.append(int(vzd) if smer == "V" else -int(vzd))

if P == 0:
    print(0)
    quit(0)

S = SortedList()
best = 0

for i in range(N):
    S.add(D[i])
    if i >= P:
        S.remove(D[i-P])
    if i >= P - 1:
        best = max(best, S.sum - 2 * S.sum_smallest_k_negatives(K))
        best = max(best, -S.sum + 2 * S.sum_largest_k_positives(K))

print(best)
